<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\User;
use App\Models\Product;
use App\Models\Category;
use App\Helpers\AuditLogger;
use App\Models\AuditLog;
use App\Services\QuickBooksService;

class AdminController extends Controller
{
    private $quickBooksService;

    public function __construct(QuickBooksService $quickBooksService)
    {
        $this->quickBooksService = $quickBooksService;
    }

    public function dashboard()
    {
        $userCount = User::count();
        $productCount = Product::count();
        $categoryCount = Category::count();

        return view('admin.dashboard', compact('userCount', 'productCount', 'categoryCount'));
    }

    public function index()
    {
        $users = User::all();
        return view('admin.users.index', compact('users'));
    }

    public function show($id)
    {
        $user = User::findOrFail($id);
        return view('admin.users.show', compact('user'));
    }

    public function edit($id)
    {
        $user = User::findOrFail($id);
        return view('admin.users.edit', compact('user'));
    }

    public function update(Request $request, $id)
    {
        $user = User::findOrFail($id);

        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|max:255|unique:users,email,' . $user->id,
            'role' => 'required|string|in:admin,seller,buyer',
        ]);

        $user->update($request->only(['name', 'email', 'role']));

        AuditLogger::log('User Updated', "Updated user with ID: {$user->id}");

        return redirect()->route('admin.users.index')->with('success', 'User updated successfully.');
    }

    public function destroy($id)
    {
        $user = User::findOrFail($id);
        $user->delete();

        return redirect()->route('admin.users.index')->with('success', 'User deleted successfully.');
    }

    public function products()
    {
        $products = Product::all();
        return view('admin.products.index', compact('products'));
    }

    public function editProduct($id)
    {
        $product = Product::findOrFail($id);
        return view('admin.products.edit', compact('product'));
    }

    public function updateProduct(Request $request, $id)
    {
        $product = Product::findOrFail($id);

        $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'required|string',
            'status' => 'required|string|in:active,inactive',
        ]);

        $product->update($request->only(['name', 'description', 'status']));

        AuditLogger::log('Product Updated', "Updated product with ID: {$product->id}");

        return redirect()->route('admin.products.index')->with('success', 'Product updated successfully.');
    }

    public function auditLogs()
    {
        $auditLogs = AuditLog::with('user')->latest()->paginate(10);
        return view('admin.audit_logs.index', compact('auditLogs'));
    }

    public function initiateQuickBooksOAuth()
    {
        $authorizationUrl = $this->quickBooksService->getAuthorizationUrl();
        return redirect($authorizationUrl);
    }

    public function handleQuickBooksCallback(Request $request)
    {
        $authorizationCode = $request->query('code');

        try {
            $tokenData = $this->quickBooksService->exchangeCodeForToken($authorizationCode);
            // Store token data securely (e.g., in the database or encrypted storage)
            return response()->json(['success' => 'QuickBooks connected successfully.', 'data' => $tokenData]);
        } catch (\Exception $e) {
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }

    public function fetchQuickBooksInventory(Request $request)
    {
        $accessToken = $request->input('access_token');
        $realmId = $request->input('realm_id');

        try {
            $inventoryItems = $this->quickBooksService->fetchInventoryItems($accessToken, $realmId);
            return response()->json(['success' => true, 'data' => $inventoryItems]);
        } catch (\Exception $e) {
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }

    public function updateQuickBooksInventory(Request $request, $itemId)
    {
        $accessToken = $request->input('access_token');
        $realmId = $request->input('realm_id');
        $data = $request->only(['name', 'quantity_on_hand', 'unit_price']);

        try {
            $updatedItem = $this->quickBooksService->updateInventoryItem($accessToken, $realmId, $itemId, $data);
            return response()->json(['success' => true, 'data' => $updatedItem]);
        } catch (\Exception $e) {
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }
}